# Flow 이미지 자동화 API

Google Flow (labs.google/fx/tools/flow) 이미지 생성을 API로 사용할 수 있게 해주는 도구입니다.

## 구성 파일

| 파일 | 설명 |
|------|------|
| `flow_api_server.py` | Python API 서버 (WebSocket 브릿지) |
| `flow_api_ws.user.js` | Tampermonkey 스크립트 |

---

## 설치 방법

### 1. Python 패키지 설치

```bash
pip install fastapi uvicorn websockets
```

### 2. Tampermonkey 스크립트 설치

1. Chrome에서 Tampermonkey 확장 프로그램 설치
2. Tampermonkey 아이콘 클릭 → "새 스크립트 만들기"
3. `flow_api_ws.user.js` 내용 전체 복사하여 붙여넣기
4. Ctrl+S (또는 Cmd+S)로 저장

### 3. 서버 실행

```bash
python3 flow_api_server.py
```

### 4. Flow 페이지 열기

1. Chrome에서 https://labs.google/fx/tools/flow 접속
2. Google 계정으로 로그인
3. 콘솔(F12)에서 "WebSocket connected" 메시지 확인

---

## API 사용법

### 엔드포인트

| 메서드 | URL | 설명 |
|--------|-----|------|
| GET | http://localhost:8002/ | 서버 상태 |
| GET | http://localhost:8002/health | 헬스체크 |
| GET | http://localhost:8002/docs | API 문서 (Swagger) |
| POST | http://localhost:8002/generate | 이미지 생성 |

---

## API 파라미터

### POST /generate

| 파라미터 | 타입 | 필수 | 기본값 | 설명 |
|----------|------|------|--------|------|
| `prompt` | string | O | - | 이미지 생성 프롬프트 |
| `count` | int | X | 2 | 이미지 개수 (1~8) |
| `model` | string | X | GEM_PIX_2 | 모델 선택 |
| `aspect_ratio` | string | X | landscape | 비율 선택 |

### 모델 (model)

| 값 | 설명 |
|----|------|
| `IMAGEN4` | Imagen 4 |
| `GEM_PIX` | Nano Banana |
| `GEM_PIX_2` | Nano Banana Pro (기본값, 추천) |

### 비율 (aspect_ratio)

| 값 | 설명 |
|----|------|
| `square` | 1:1 정사각형 |
| `landscape` | 16:9 가로 (기본값) |
| `portrait` | 9:16 세로 |

---

## 명령어 예시

### curl

```bash
# 기본 (2장, landscape)
curl -X POST http://localhost:8002/generate -H "Content-Type: application/json" -d '{"prompt": "a beautiful sunset"}'

# 4장, 정사각형
curl -X POST http://localhost:8002/generate -H "Content-Type: application/json" -d '{"prompt": "a cute cat", "count": 4, "aspect_ratio": "square"}'

# Imagen4 모델, 세로
curl -X POST http://localhost:8002/generate -H "Content-Type: application/json" -d '{"prompt": "a tall building", "model": "IMAGEN4", "aspect_ratio": "portrait"}'

# 전체 옵션
curl -X POST http://localhost:8002/generate -H "Content-Type: application/json" -d '{"prompt": "a robot", "count": 4, "model": "GEM_PIX_2", "aspect_ratio": "square"}'
```

### Python

```python
import requests

response = requests.post("http://localhost:8002/generate", json={
    "prompt": "a magical forest",
    "count": 4,
    "model": "GEM_PIX_2",
    "aspect_ratio": "square"
})

result = response.json()
print(result)

# 이미지 URL 출력
if result["success"]:
    for img in result["images"]:
        print(img["url"])
```

### JavaScript (Node.js)

```javascript
const response = await fetch("http://localhost:8002/generate", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({
        prompt: "a futuristic city",
        count: 2,
        model: "GEM_PIX_2",
        aspect_ratio: "landscape"
    })
});

const result = await response.json();
console.log(result);
```

---

## 응답 형식

### 성공

```json
{
    "success": true,
    "count": 2,
    "images": [
        {
            "url": "https://storage.googleapis.com/...",
            "prompt": "a beautiful sunset",
            "seed": 123456
        },
        {
            "url": "https://storage.googleapis.com/...",
            "prompt": "a beautiful sunset",
            "seed": 789012
        }
    ],
    "error": null
}
```

### 실패

```json
{
    "success": false,
    "count": null,
    "images": [],
    "error": "에러 메시지"
}
```

---

## 문제 해결

### "브라우저가 연결되지 않았습니다" 오류

1. Flow 페이지가 열려있는지 확인
2. Google 계정으로 로그인되어 있는지 확인
3. 콘솔(F12)에서 "WebSocket connected" 메시지 확인
4. 페이지 새로고침 후 다시 시도

### "Timeout" 오류

- 이미지 생성에 시간이 오래 걸릴 수 있음 (최대 60초)
- Flow 페이지가 정상 작동하는지 확인

### Tampermonkey 스크립트가 작동하지 않음

1. Tampermonkey가 활성화되어 있는지 확인
2. 스크립트가 활성화되어 있는지 확인
3. Flow 페이지 새로고침
4. 콘솔에서 "[FlowAPI]" 로그 확인

---

## 아키텍처

```
┌─────────────────┐     HTTP      ┌─────────────────┐
│   외부 앱       │ ────────────> │  Python 서버    │
│ (curl, Python)  │    :8002      │  (FastAPI)      │
└─────────────────┘               └────────┬────────┘
                                           │
                                      WebSocket
                                           │
                                  ┌────────▼────────┐
                                  │   Chrome        │
                                  │  Tampermonkey   │
                                  │    + Flow UI    │
                                  └─────────────────┘
```
